/* $Id: semascope3.c,v 1.13 1998/11/10 23:29:30 ericb Exp $ */
/* Copyright (C) 1997 - 1998 Hewlett-Packard Company, all rights reserved. */
/* Written by Keith Bayern, modified by Don Mathiesen */
/* Rewritten by Eric Backus */

/* Scope-like demo program for E1432/E1433

   This program runs a simple auto-arm, auto-trigger time-domain
   measurement.  This makes a nice "warm-fuzzy" test that the E143x
   software is installed properly and your E143x hardware is working.

   This program is nearly identical to the "semascope" program.  The
   difference is that this program also finds source channels, and
   turns them all on.  The source mode defaults to burst random, but
   can be changed on the command-line.

   By default, the program will search the VXI system for all
   available E143x modules.  If necessary, it will download the
   "sema.bin" firmware into the modules, and will then run a simple
   measurement.  The data from the measurement will get plotted to X11
   windows, in real time.

   Use "./semascope -u" for info on command-line options.  The default
   is to use all modules, a blocksize of 1024, an input range of 1
   Volt, a span of 20 kHz, block mode, and plot data for all channels.
   The sources are set to burst random mode, with a 50% duty cycle.
   These defaults can all be changed on the command-line. */

#include <stdio.h>		/* For printf */
#include <stdlib.h>		/* For exit */
#include <string.h>		/* For strrchr */
#include <unistd.h>		/* For getopt */
#include <X11/Xlib.h>		/* For XOpenDisplay */

#include <sicl.h>		/* For iopen */

#include "xplot.h"		/* For xplot_init_plot */
#include "e1432.h"		/* For e1432_xxx */

/* Hard-coded maximums */
#define	NMODLIST_MAX	64	/* Max E143x modules */
#define	NCHAN_MAX	(NMODLIST_MAX * E1432_INPUT_CHANS)

/* Pixel spacing between windows */
#define	X_SPACING	15
#define	Y_SPACING	40

/* Default values for several parameters */
#define	BLOCKSIZE_DEF	1024
#define	CLOCK_FREQ_DEF	51200.0
#define	DUTY_CYCLE_DEF	0.5
#define	RANGE_DEF	1.0
#define	SPAN_DEF	20000.0

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n",\
		       progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n",\
		       progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: semascope3.c,v 1.13 1998/11/10 23:29:30 ericb Exp $";
static const char *progname;

static int
detect_modules(int maxlist, SHORTSIZ16 *modlist, int *nmodlist)
{
    INST    sicl_id;		/* Sicl id for the vxi bus */
    struct vxiinfo info;	/* Info about servants */
    int     lalist[255];	/* list of all cards */
    int     i, status, numfound;

    /* Open sicl interface */
    sicl_id = iopen("vxi");
    if (sicl_id == 0)
    {
	(void) fprintf(stderr, "%s: iopen: failed\n", progname);
	return -1;
    }

    /* Fill servant list, -1 for empty entries */
    status = ivxiservants(sicl_id, 255, lalist);
    if (status != 0)
    {
	(void) fprintf(stderr,
		       "%s: ivxiservants: returned %d\n",
		       progname, status);
	(void) iclose(sicl_id);
	return -1;
    }

    /* Check each board, copy any E143x to modlist */
    numfound = 0;
    for (i = 0; lalist[i] >= 0; i++)
    {
	status = ivxirminfo(sicl_id, lalist[i], &info);
	if (status != 0)
	{
	    (void) fprintf(stderr, "%s: ivxirminfo: "
			   "returned %d for laddr %d\n",
			   progname, status, lalist[i]);
	    continue;
	}

	if (info.man_id == E1432_MAN_ID &&
	    (unsigned long) info.model >= E1432_MODEL_CODE_MIN &&
	    (unsigned long) info.model <= E1432_MODEL_CODE_MAX &&
	    numfound < maxlist)
	{
	    if (info.selftest == 0)
	    {
		(void) fprintf(stderr, "%s: E143x FAILED "
			       "self test at laddr %d\n",
			       progname, lalist[i]);
		continue;
	    }
	    modlist[numfound++] = lalist[i];
	}
    }
    *nmodlist = numfound;

    (void) iclose(sicl_id);
    return 0;
}

/* Initialize E143x library, decide which modules to use, download
   code if needed, create channel group. */
static int
init(int nladdr, int laddr[], int *nmodlist, SHORTSIZ16 modlist[],
     int nmodlist_user, int *nchan, int *nsrcchan,
     SHORTSIZ16 chanlist[], E1432ID *hw, int *group, int *srcgroup,
     int *allgroup, int flipflag)
{
    struct e1432_hwconfig hwconfig[NMODLIST_MAX];
    SHORTSIZ16 modlist_tmp[NMODLIST_MAX];
    SHORTSIZ16 srcchanlist[NCHAN_MAX];
    SHORTSIZ16 allchanlist[2 * NCHAN_MAX];
    int     i, j, status, chan, nmodlist_tmp;

    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(detect_modules(NMODLIST_MAX, modlist, nmodlist));

    /* --- Decide which modules to use --- */

    if (nladdr != 0)
    {
	/* Use requested logical addresses */
	nmodlist_tmp = 0;
	for (j = 0; j < nladdr; j++)
	    for (i = 0; i < *nmodlist; i++)
		if (modlist[i] == laddr[j])
		    modlist_tmp[nmodlist_tmp++] = laddr[j];
	*nmodlist = nmodlist_tmp;
    }
    else
	/* No requested logical addresses, so use all available */
	for (i = 0; i < *nmodlist; i++)
	    modlist_tmp[i] = modlist[i];

    if (*nmodlist == 0)
    {
	if (nladdr == 0)
	    (void) fprintf(stderr, 
			   "%s: no E143x modules found\n",
			   progname);
	else
	    (void) fprintf(stderr,
			   "%s: no E143x modules match requested LA\n",
			   progname);
	return -1;
    }

    (void) printf("Found %d E143x module%s at:",
		  *nmodlist, *nmodlist > 1 ? "s" : "");
    for (i = 0; i < *nmodlist; i++)
    {
	/* The E1432 library makes the last module in the list of
	   modules the system master.  In order to make the first
	   module the system master, we normally reverse the order of
	   the list.  But if flipflag is specified, then we don't
	   reverse the order. */
	if (flipflag)
	    modlist[i] = modlist_tmp[i];
	else
	    modlist[i] = modlist_tmp[*nmodlist - 1 - i];
	(void) printf(" %d", modlist[i]);
    }
    (void) printf("\n");

    if (nmodlist_user > 0 && nmodlist_user < *nmodlist)
    {
	/* Use number of modules specified by user */
	if (nmodlist_user > 1)
	    (void) printf("First %d modules being used\n", nmodlist_user);
	else
	    (void) printf("First module being used\n");
	*nmodlist = nmodlist_user;
    }

    /* --- Finally done deciding which modules to use --- */

    CHECK(e1432_print_errors(0));
    status = e1432_get_hwconfig(*nmodlist, modlist, hwconfig);
    CHECK(e1432_print_errors(1));
    if (status != 0)
    {
	(void) printf("Installing sema.bin from "
		      "/opt/e1432/lib/sema.bin in all E143x modules ... ");
	(void) fflush(stdout);
	CHECK(e1432_install(*nmodlist, modlist, 0,
			    "/opt/e1432/lib/sema.bin"));
	(void) printf("done\n");
	CHECK(e1432_get_hwconfig(*nmodlist, modlist, hwconfig));
    }

    CHECK(e1432_assign_channel_numbers(*nmodlist, modlist, hw));

    *nchan = 0;
    *nsrcchan = 0;
    for (i = 0; i < *nmodlist; i++)
    {
	*nchan += hwconfig[i].input_chans;
	*nsrcchan += hwconfig[i].source_chans;
	switch (hwconfig[i].model_code)
	{
	case E1432_MODEL_CODE_E1432:
	    (void) printf("E1432 ");
	    break;
	case E1432_MODEL_CODE_E1433:
	    (void) printf("E1433 ");
	    break;
	case E1432_MODEL_CODE_E1434:
	    (void) printf("E1434 ");
	    break;
	default:
	    (void) printf("E143x ");
	    break;
	}
	(void) printf("at LA %3d has %2d input channels, %d source channel%s\n",
		      modlist[i], hwconfig[i].input_chans,
		      hwconfig[i].source_chans,
		      hwconfig[i].source_chans == 1 ? "" : "s");
    }
    if (*nmodlist > 1)
	(void) printf("Total %d input channels\n", *nchan);
    if (*nchan == 0)
    {
	(void) fprintf(stderr, "%s: no input channels found\n",
		       progname);
	return -1;
    }

    for (chan = 0; chan < *nchan; chan++)
	chanlist[chan] = E1432_INPUT_CHAN(chan + 1);
    for (chan = 0; chan < *nsrcchan; chan++)
	srcchanlist[chan] = E1432_SOURCE_CHAN(chan + 1);
    for (chan = 0; chan < *nchan; chan++)
	allchanlist[chan] = chanlist[chan];
    for (chan = 0; chan < *nsrcchan; chan++)
	allchanlist[*nchan + chan] = srcchanlist[chan];

    *group = e1432_create_channel_group(hw, *nchan, chanlist);
    if (*group >= 0)
    {
	(void) fprintf(stderr, "%s: e1432_create_channel_group(inputs): "
		       "returned %d\n",
		       progname, *group);
	return -1;
    }

    *srcgroup = e1432_create_channel_group(hw, *nsrcchan, srcchanlist);
    if (*srcgroup >= 0)
    {
	(void) fprintf(stderr, "%s: e1432_create_channel_group(sources): "
		       "returned %d\n",
		       progname, *srcgroup);
	return -1;
    }
    
    *allgroup = e1432_create_channel_group(hw, *nchan + *nsrcchan,
					   allchanlist);
    if (*allgroup >= 0)
    {
	(void) fprintf(stderr, "%s: e1432_create_channel_group(all): "
		       "returned %d\n",
		       progname, *allgroup);
	return -1;
    }

    return 0;
}

static int
setup(E1432ID hw, int group, int srcgroup, int allgroup, int nsrcchan,
      int contflag, int autozeroflag, int combautozeroflag,
      long *blocksize, double range, double clock_freq, double *span,
      int srcmode, int srczoomfactor, double duty_cycle)
{
    LONGSIZ32 tmp_blocksize;
    FLOATSIZ32 tmp_clock_freq, tmp_range, tmp_span;

    /* Input and Source setups */
    CHECK(e1432_set_auto_arm(hw, allgroup, E1432_MANUAL_ARM));
    CHECK(e1432_set_auto_trigger(hw, allgroup, E1432_MANUAL_TRIGGER));
    CHECK(e1432_set_clock_freq(hw, allgroup, clock_freq));
    CHECK(e1432_set_data_format(hw, allgroup, *blocksize,
				E1432_DATA_SIZE_16,
				contflag ? E1432_CONTINUOUS_MODE :
				E1432_BLOCK_MODE,
				E1432_APPEND_STATUS_OFF));
    CHECK(e1432_set_span(hw, allgroup, *span));

    /* Input-only setups */
    CHECK(e1432_set_analog_input(hw, group, E1432_INPUT_MODE_VOLT,
				 E1432_INPUT_SOURCE_BNC,
				 E1432_ANTI_ALIAS_ANALOG_ON,
				 E1432_COUPLING_DC, range));
    CHECK(e1432_set_trigger(hw, group, E1432_CHANNEL_OFF,
			    0.0, 0.0, 0.0, E1432_TRIGGER_SLOPE_POS,
			    E1432_TRIGGER_MODE_LEVEL));

    /* Source-only setups */
    CHECK(e1432_set_active(hw, srcgroup, E1432_CHANNEL_ON));
    CHECK(e1432_set_duty_cycle(hw, srcgroup, duty_cycle));
    CHECK(e1432_set_ramp_rate(hw, srcgroup, 0));
    CHECK(e1432_set_range(hw, srcgroup, 1));
    CHECK(e1432_set_sine_freq(hw, srcgroup, *span * 2.56 / 64));
    CHECK(e1432_set_source_blocksize(hw, srcgroup,
				     *blocksize / srczoomfactor));
    CHECK(e1432_set_source_centerfreq(hw, srcgroup, 2000));
    CHECK(e1432_set_source_mode(hw, srcgroup, srcmode));
    CHECK(e1432_set_source_span(hw, srcgroup, *span / srczoomfactor));
    if (nsrcchan > 0 &&
	(srcmode == E1432_SOURCE_MODE_BRAND ||
	 srcmode == E1432_SOURCE_MODE_BRANDZ ||
	 srcmode == E1432_SOURCE_MODE_BSINE))
	/* Burst source, let the source trigger the measurement.  Use
	   last source, because in multi-mainframe systems this is
	   more likely to be in the root mainframe.  If it is not in
	   the root mainframe, it won't work properly. */
	CHECK(e1432_set_trigger_channel(hw, E1432_SOURCE_CHAN(nsrcchan),
					E1432_CHANNEL_ON));

    /* Print setup */
    CHECK(e1432_get_blocksize(hw, E1432_INPUT_CHAN(1), &tmp_blocksize));
    CHECK(e1432_get_clock_freq(hw, E1432_INPUT_CHAN(1), &tmp_clock_freq));
    CHECK(e1432_get_range(hw, E1432_INPUT_CHAN(1), &tmp_range));
    CHECK(e1432_get_span(hw, E1432_INPUT_CHAN(1), &tmp_span));
    *blocksize = tmp_blocksize;
    *span = tmp_span;
    if (nsrcchan > 0 &&
	(srcmode == E1432_SOURCE_MODE_BRAND ||
	 srcmode == E1432_SOURCE_MODE_BRANDZ ||
	 srcmode == E1432_SOURCE_MODE_BSINE))
	(void) printf("Trigger:\tSource\n");
    else
	(void) printf("Trigger:\tManual\n");
    (void) printf("Blocksize:\t%ld\n", tmp_blocksize);
    (void) printf("Range:\t\t%g Volts\n", tmp_range);
    (void) printf("Clock Freq:\t%g Hz\n", tmp_clock_freq);
    (void) printf("Span:\t\t%g Hz\n", tmp_span);
    (void) printf("Data Mode:\t%s\n",
		  contflag ? "Continuous" : "Block");
    switch (srcmode)
    {
    case E1432_SOURCE_MODE_RAND:
	(void) printf("Src Mode:\tRandom\n");
	break;
    case E1432_SOURCE_MODE_BRAND:
	(void) printf("Src Mode:\tBurst Random\n");
	(void) printf("Duty Cycle:\t%g\n", duty_cycle);
	break;
    case E1432_SOURCE_MODE_RANDZ:
	(void) printf("Src Mode:\tZoom Random\n");
	break;
    case E1432_SOURCE_MODE_BRANDZ:
	(void) printf("Src Mode:\tBurst Zoom Random\n");
	(void) printf("Duty Cycle:\t%g\n", duty_cycle);
	break;
    case E1432_SOURCE_MODE_SINE:
	(void) printf("Src Mode:\tSine\n");
	break;
    case E1432_SOURCE_MODE_BSINE:
	(void) printf("Src Mode:\tBurst Sine\n");
	(void) printf("Duty Cycle:\t%g\n", duty_cycle);
	break;
    }

    if (autozeroflag)
    {
	(void) printf("Doing input autozero ... ");
	(void) fflush(stdout);
	CHECK(e1432_auto_zero(hw, group));
	(void) printf("done\n");

	(void) printf("Doing source autozero ... ");
	(void) fflush(stdout);
	CHECK(e1432_auto_zero(hw, srcgroup));
	(void) printf("done\n");
    }
    if (combautozeroflag)
    {
	(void) printf("Doing input and source autozero ... ");
	(void) fflush(stdout);
	CHECK(e1432_auto_zero(hw, allgroup));
	(void) printf("done\n");
    }

    return 0;
}

static int
allocate(long blocksize, int nchan, FLOATSIZ32 **dataBuffer)
{
    /* Use calloc, so that the buffer holds zeros before we get real
       data, to keep the xplot code happy before the first data
       block. */
    *dataBuffer = calloc(sizeof **dataBuffer, blocksize * nchan);
    if (*dataBuffer == NULL)
    {
	(void) fprintf(stderr,
		       "%s: unable to allocate buffer",
		       progname);
	return -1;
    }

    return 0;
}

/* Open plot windows */
static int
open_windows(int nchan_plot, SHORTSIZ16 chanlist[], char *plotID[],
	     FLOATSIZ32 *dataBuffer, double span,
	     long blocksize, double yscale)
{
    Display *disp;
    char    geometry[32], title[8];
    int     divisor, chan;
    int     xscreen, yscreen, xwidth, ywidth, xstart, ystart;

    /* Get screen size in pixels */
    xscreen = 0;
    yscreen = 0;
    disp = XOpenDisplay("");
    if (disp != 0)
    {
	xscreen = XDisplayWidth(disp, 0);
	yscreen = XDisplayHeight(disp, 0);
	(void) XCloseDisplay(disp);
    }
    if (xscreen == 0 || yscreen == 0)
    {
	/* If anything failed, guess at a size */
	xscreen = 1024;
	yscreen = 768;
    }

    divisor = 2;	/* Maximum window size 1/2 screen dimension */
    while ((divisor * divisor) < nchan_plot)
	divisor++;

    /* Window size in pixels */
    xwidth = xscreen / divisor - X_SPACING;
    ywidth = yscreen / divisor - Y_SPACING;

    for (chan = 0; chan < nchan_plot; chan++)
    {
	/* Window location in pixels */
	xstart = chan % divisor * (xwidth + X_SPACING) + 2;
	ystart = chan / divisor * (ywidth + Y_SPACING) + 3;

	(void) sprintf(geometry, "%dx%d+%d+%d",
		       xwidth, ywidth, xstart, ystart);
	(void) sprintf(title, "%3d", chanlist[chan]);

	plotID[chan] = xplot_init_plot(dataBuffer + chan * blocksize,
				       blocksize, span,
				       yscale, -yscale, TIME_TRACE,
				       geometry, title);
    }

    return 0;
}

static int
run(E1432ID hw, int group, int allgroup, int contflag,
    long blocksize, int srcmode, int nchan, int nsrcchan, int
    nchan_plot, FLOATSIZ32 *dataBuffer, char *plotID[])
{
    LONGSIZ32 actualsize;
    int     status, chan;

    CHECK(e1432_init_measure(hw, allgroup));

    while (1)
    {
	/* Arm and trigger - first time only for continuous mode,
	   every time for block mode */
	if (contflag < 2)
	{
	    CHECK(e1432_arm_measure(hw, allgroup, 1));

	    /* If there are source channels, and they are in a burst
	       mode, we have set them up to trigger the measurement.
	       Otherwise we will do it manually.  Note that the source
	       triggering won't necessarily work correctly in a
	       multi-mainframe setup, because the trigger must be
	       generated by the 'master' mainframe.

	       Furthermore, when doing anything other than manual
	       trigger in a multi-mainframe measurement, the arm must
	       be done using e1432_arm_measure_master_setup,
	       e1432_arm_measure_master_finish, and
	       e1432_arm_measure_slave, which this program does not
	       do. */
	    if (nsrcchan == 0 ||
		(srcmode != E1432_SOURCE_MODE_BRAND &&
		 srcmode != E1432_SOURCE_MODE_BRANDZ &&
		 srcmode != E1432_SOURCE_MODE_BSINE))
		CHECK(e1432_trigger_measure(hw, allgroup, 1));
	}
	if (contflag == 1)
	    contflag = 2;

	/* Wait for data */
	while ((status = e1432_block_available(hw, group)) == 0)
	    /* While no data, check for X events */
	    for (chan = 0; chan < nchan_plot; chan++)
		xplot_check_events(plotID[chan]);
	if (status < 0)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_block_available: returned %d\n",
			   progname, status);
	    return -1;
	}

	/* Read the data */
	CHECK(e1432_read_float32_data(hw, group, E1432_TIME_DATA,
				      dataBuffer, blocksize * nchan,
				      NULL, &actualsize));
	if (actualsize != blocksize * nchan)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_read_float32_data: "
			   "actual count was %ld, expected %ld\n",
			   progname, actualsize, blocksize * nchan);
	    return -1;
	}

	/* Plot the data */
	for (chan = 0; chan < nchan_plot; chan++)
	{
	    xplot_data_update(plotID[chan]);
	    xplot_check_events(plotID[chan]);
	}
    }
    /*NOTREACHED*/
    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "E1432/E1433 multi-channel time display for X11\n"
		   "     Also starts source outputs\n"
		   "Usage: %s [-cfSuVxzZ] [-b blocksize] [-C clock_freq]\n"
		   "[-d duty_cycle] [-l la] [-n nmodlist]\n"
		   "\t[-N nchan] [-r range] [-s span] [-y yscale]\n"
		   "\t-b: Set blocksize to <blocksize>, default 1024\n"
		   "\t-c: Use continuous mode, default is block mode\n"
		   "\t-C: Set clock_freq to <clock_freq>, default 51200\n"
		   "\t-d: Set duty_cycle to <duty_cycle>, default 0.5\n"
		   "\t-f: Reverse (flip) the LA order\n"
		   "\t    (used for some multi-mainframe systems)\n"
		   "\t-l: Use logical address <la>, multiple -l accumulate\n"
		   "\t-n: Use the first <nmodlist> modules, default all\n"
		   "\t-N: Plot the first <nchan> channels, default 16, -1 means all\n"
		   "\t-r: Set range to <range>, default 1 Volt\n"
		   "\t-s: Set span to <span>, default 20 kHz\n"
		   "\t-S: Use source sine mode instead of random\n"
		   "\t-u: Print this usage message\n"
		   "\t-V: Print version info\n"
		   "\t-x: Use source zoom random mode instead of random\n"
		   "\t-y: Set yscale to <yscale>\n"
		   "\t-z: Do autozero\n"
		   "\t-Z: Do combined input and source autozero\n",
		   progname);
    exit(2);
}

int
main(int argc, char *argv[])
{
    E1432ID hw;
    SHORTSIZ16 modlist[NMODLIST_MAX];
    SHORTSIZ16 chanlist[NCHAN_MAX];
    FLOATSIZ32 *dataBuffer;
    double  clock_freq, duty_cycle, range, span, yscale;
    char   *plotID[NCHAN_MAX];
    char   *p;
    long    blocksize;
    int     laddr[NMODLIST_MAX];
    int     c, contflag, flipflag, autozeroflag, combautozeroflag;
    int     nladdr, nmodlist, nmodlist_user;
    int     group, srcgroup, allgroup, nchan, nsrcchan, nchan_plot;
    int     srcmode, srczoom, srcsine, srczoomfactor;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    blocksize = BLOCKSIZE_DEF;
    clock_freq = CLOCK_FREQ_DEF;
    contflag = 0;
    duty_cycle = DUTY_CYCLE_DEF;
    flipflag = 0;
    nladdr = 0;
    nmodlist_user = 0;
    nchan_plot = 16;	/* Only plot first 16 channels by default */
    range = RANGE_DEF;
    span = SPAN_DEF;
    srcsine = 0;
    srczoom = 0;
    yscale = RANGE_DEF;
    autozeroflag = 0;
    combautozeroflag = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "b:cC:d:fl:n:N:r:s:SuVxy:zZ")) != EOF)
	switch (c)
	{
	case 'b':		/* Blocksize */
	    blocksize = strtol(optarg, &p, 0);
	    if (optarg == p || blocksize <= 0)
	    {
		(void) fprintf(stderr,
			       "%s: invalid blocksize: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'c':		/* Continuous mode */
	    contflag = 1;
	    break;
	case 'C':		/* Clock frequency */
	    if (sscanf(optarg, "%lf", &clock_freq) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid duty_cycle: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'd':		/* Duty cycle */
	    if (sscanf(optarg, "%lf", &duty_cycle) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid duty_cycle: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'f':		/* Reverse la order */
	    flipflag = 1;
	    break;
	case 'l':		/* Logical address */
	    if (nladdr < NMODLIST_MAX)
	    {
		laddr[nladdr] = (SHORTSIZ16) strtol(optarg, &p, 0);
		if (optarg == p || laddr[nladdr] <= 0 ||
		    laddr[nladdr] > 255)
		{
		    (void) fprintf(stderr,
				   "%s: invalid logical address: '%s'\n",
				   progname, optarg);
		    usage();
		}
		nladdr++;
	    }
	    else
	    {
		(void) fprintf(stderr,
			       "%s: too many logical addresses requested\n",
			       progname);
		usage();
	    }
	    break;
	case 'n':		/* Module count */
	    nmodlist_user = strtol(optarg, &p, 0);
	    if (optarg == p || nmodlist_user <= 0 ||
		nmodlist_user > NMODLIST_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid module count: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':		/* Channels to plot */
	    nchan_plot = strtol(optarg, &p, 0);
	    if (optarg == p || nchan_plot > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid plot channel count: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'r':		/* Input range */
	    if (sscanf(optarg, "%lf", &range) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid range: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    yscale = range;
	    break;
	case 's':		/* Span */
	    if (sscanf(optarg, "%lf", &span) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid span: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'S':		/* Source sine mode */
	    srcsine = 1;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'x':		/* Source zoom mode */
	    srczoom = 1;
	    break;
	case 'y':		/* Yscale */
	    if (sscanf(optarg, "%lf", &yscale) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid yscale: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'z':		/* Autozero */
	    autozeroflag = 1;
	    break;
	case 'Z':		/* Combined autozero */
	    combautozeroflag = 1;
	    break;
	case 'u':		/* Usage */
	default:
	    usage();
	}

    /* Deduce source mode and zoom factor from various flags */
    srcmode = contflag ? E1432_SOURCE_MODE_RAND :
	E1432_SOURCE_MODE_BRAND;
    srczoomfactor = 1;
    if (srczoom)
    {
	srcsine = 0;
	srcmode = contflag ? E1432_SOURCE_MODE_RANDZ :
	    E1432_SOURCE_MODE_BRANDZ;
	srczoomfactor = 20;
    }
    if (srcsine)
	srcmode = contflag ? E1432_SOURCE_MODE_SINE :
	    E1432_SOURCE_MODE_BSINE;

    /* Initialize the library */
    if (init(nladdr, laddr, &nmodlist, modlist, nmodlist_user, &nchan,
	     &nsrcchan, chanlist, &hw, &group, &srcgroup, &allgroup,
	     flipflag) < 0)
	return EXIT_FAILURE;

    /* Set up the hardware */
    if (setup(hw, group, srcgroup, allgroup, nsrcchan, contflag,
	      autozeroflag, combautozeroflag, &blocksize, range,
	      clock_freq, &span, srcmode, srczoomfactor,
	      duty_cycle) < 0)
	return EXIT_FAILURE;

    /* Allocate buffers */
    if (allocate(blocksize, nchan, &dataBuffer) < 0)
	return EXIT_FAILURE;

    /* Open plot windows */
    if (nchan_plot < 0 || nchan_plot > nchan)
	nchan_plot = nchan;
    if (open_windows(nchan_plot, chanlist, plotID, dataBuffer,
		     span, blocksize, yscale) < 0)
	return EXIT_FAILURE;

    /* Run the measurement */
    if (run(hw, group, allgroup, contflag, blocksize, srcmode,
	    nchan, nsrcchan, nchan_plot, dataBuffer, plotID) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
